﻿using System;
using System.Collections.Generic;
using System.Linq;
using BMS.Facade;
using BMS.Utils;
using BMS.VistaIntegration.Data;
using VI = BMS.VistaIntegration.FacadeContracts;
using BMS.VistaIntegration.Translators;
using BMS.VistaWorker2.Writer;
using BMS.VistaWorker2.Writer.Implementation;
using BMS.VistaIntegration.VistA;
using BMS.VistaIntegration.Mdws;
using BMS.VistaIntegration.Cache;
using BMS.VistaIntegration.HL7.DataContracts;
using System.Xml.Serialization;
using System.Text;
using System.IO;
using InfoWorld.HL7.ITS;
using System.Configuration;

namespace BMS.VistaIntegration.HL7.ServiceImplementation
{
    public abstract class HL7OperationsBase
    {
        private Dictionary<int, IWriterManager> writerManagers;
        public Dictionary<int, IWriterManager> WriterManagers
        {
            get
            {
                if (writerManagers == null)
                    writerManagers = new Dictionary<int, IWriterManager>();
                return writerManagers;
            }
        }

        private Dictionary<int, VI.VistASite> siteConfigurations;
        public Dictionary<int, VI.VistASite> SiteConfigurations
        {
            get
            {
                if (siteConfigurations == null)
                    siteConfigurations = new Dictionary<int, VI.VistASite>();
                return siteConfigurations;
            }
        }

        public IVistASession GetVistASession(VI.Admin.VistaDataType dataType, string facilityIdentifier)
        {
            VI.VistASite vistaConfig = GetSiteConfiguration(facilityIdentifier);
            if (vistaConfig == null)
                return null;


            if (!vistaConfig.DataRetrievalDetails.ContainsKey(dataType))
            {
                Tracer.TraceMessage("Received an event from the VistA with number " + facilityIdentifier +
                    ". The message was not processed because no appropriate configuration (" + dataType.ToString("G") + ") for this VistA was found.");
                return null;
            }

            VI.Admin.DataRetrieval dataRetrieval = vistaConfig.DataRetrievalDetails[dataType];

            IVistASessionFactory factory = null;
            VistAConnectionInfo connection = null;
            Facade.Data.VistaSite vistaSiteEis = BMS.ServicesWrapper.EIS.EISFactory.InstanceFromWCF.GetVistaSite(new II(BMS.ServicesWrapper.Security.SecurityFactory.InstanceWindows.GetCurrentDomain(), vistaConfig.Id));            
            var site = new VistASite(vistaConfig.Id, vistaConfig.Name, vistaSiteEis.Number, vistaConfig.TimeZone, ConfigurationManager.AppSettings[vistaConfig.MdwsEndpointConfigKey]);

            if (dataRetrieval.DataRetrievalMethod == DataRetrievalMethod.MDWS && dataRetrieval.IsHL7Enabled)
            {
                factory = new MdwsVistASessionFactory();
                connection = null;
            }
            else if (dataRetrieval.DataRetrievalMethod == DataRetrievalMethod.ODBC && dataRetrieval.IsHL7Enabled)
            {
                factory = new CacheVistASessionFactory();
                connection = new VistAConnectionInfo(
                    vistaConfig.OdbcConnectionString,
                    vistaConfig.OdbcUser,
                    vistaConfig.OdbcPwd);
            }

            if (factory == null)
                return null;

            IVistASession vistaSession = factory.MakeVistASession(site);
            vistaSession.Open(connection);

            return vistaSession;
        }

        public VI.VistASite GetSiteConfiguration(string facilityIdentifier)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                int siteId = 0;
                bool success = Int32.TryParse(facilityIdentifier, out siteId);
                //vista only accepts integer site ids, but we have to make sure
                if (!success)
                    return null;
                //check for the appropriate config
                if (SiteConfigurations.ContainsKey(siteId))
                    return SiteConfigurations[siteId];
                else
                {
                    string domain = BMS.ServicesWrapper.Security.SecurityFactory.InstanceWindows.GetCurrentDomain();
                    VI.VistASite configSite = FacadeManager.ConfigurationInterface.GetVistaSites(domain).FirstOrDefault(vs => vs.Number == siteId + "");
                    if (configSite == null)
                    {
                        Tracer.TraceMessage("Received an ADT event from the VistA with number " + siteId + ". The message was not processed because no configuration for this VistA was found.");
                        return null;
                    }
                    else
                    {
                        SiteConfigurations.Add(siteId, configSite);
                        return configSite;
                    }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IWriterManager GetWriterManager(string facilityIdentifier)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                int siteId = 0;
                bool success = Int32.TryParse(facilityIdentifier, out siteId);
                //vista only accepts integer site ids, but we have to make sure
                if (!success)
                    return null;
                IWriterManager writerManager = null;
                //check for the appropriate writer manager
                if (WriterManagers.ContainsKey(siteId))
                    writerManager = WriterManagers[siteId];
                else
                {
                    writerManager = new WriterManagerFactory().MakeWriter();
                    VI.VistASite configSite = GetSiteConfiguration(facilityIdentifier);
                    if (configSite == null)
                        return null;
                    VistASite vistaSite = new VistASite(configSite.Id, configSite.Name, configSite.Number, configSite.TimeZone, ConfigurationManager.AppSettings[configSite.MdwsEndpointConfigKey]);
                    writerManager.Open(vistaSite);
                    WriterManagers.Add(siteId, writerManager);
                }

                return writerManager;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public string GetHL7EventString(HL7Event hl7Event)
        {

            StringBuilder sb = new StringBuilder();
            using (TextWriter tw = new StringWriter(sb))
            {
                XmlSerializer serializer = new XmlSerializer(typeof(HL7Event));
                serializer.Serialize(tw, hl7Event);
                return sb.ToString();
            }
        }
    }
}
